import request from 'supertest';
import { Test, TestingModule } from '@nestjs/testing';
import { AuthService } from '../../../tm-public/src/authentication/auth.service';
import { HttpException } from '@nestjs/common';
import { HttpStatus, INestApplication } from '@nestjs/common';
import { getSigners } from '../utils/setup';
import { AppModule as publicApp } from '../../../tm-public/src/app.module';
import { AuthGuard } from '../../../tm-public/src/guards/auth.guard';
import { Reflector } from '@nestjs/core';

let app: INestApplication;
let signerWithAddress;

//JWT_SECRET=sweet-salt
//Payload
// {
//   sub: '1234567890',
//   name: 'John Doe',
//   iat: 1516239022,
// }
const validJwtToken =
  'Bearer eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJzdWIiOiIxMjM0NTY3ODkwIiwibmFtZSI6IkpvaG4gRG9lIiwiaWF0IjoxNTE2MjM5MDIyfQ.SUkkj1r5ncz9moS4OpX1bJfXPMiJviXdoFOTfQh93JE';
const invalidJwtToken =
  'Bearer eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJzdWIiOiIxMjM0NTY3ODkwIiwibmFtZSI6IkpvaG4gRG9lIiwiaWF0IjoxNTE2MjM5MDIyfQ.SUkkj1r5ncz9moS4OpX1bJfXPMiJviXdoFOTfQh9zzz';

describe('Authentication tests', () => {
  let authService: AuthService;
  let reflector: Reflector;
  let endpoints = [];

  beforeAll(async () => {
    const module: TestingModule = await Test.createTestingModule({
      providers: [AuthService],
    }).compile();

    authService = module.get<AuthService>(AuthService);
    reflector = module.get<Reflector>(Reflector);

    const moduleRef = await Test.createTestingModule({
      imports: [publicApp],
    }).compile();

    app = moduleRef.createNestApplication();
    app.useGlobalGuards(new AuthGuard(authService, reflector));
    await app.init();

    signerWithAddress = await getSigners();

    endpoints = [
      {
        method: 'post',
        url: '/submissions/order',
        body: { oid: 'sub1' },
      },
      {
        method: 'get',
        url: '/payment/get-payment-allowance-on-operator',
        query: { userAddress: signerWithAddress[1].address },
      },
      {
        method: 'get',
        url: `/contract-info/payment-balances/accounts/${signerWithAddress[1].address}`,
      },
    ];
  });

  it('should decode token successfully', async () => {
    const request: any = {
      headers: {
        Authorization: validJwtToken,
      },
    };

    const result = await authService.getJwtTokenPayloadFromRequest(request);
    expect(result).toEqual({
      userDetails: {
        sub: '1234567890',
        name: 'John Doe',
        iat: 1516239022,
      },
    });
  });

  it('should throw an error for invalid token structure', async () => {
    const request: any = { headers: { Authorization: 'InvalidToken' } };

    await expect(
      authService.getJwtTokenPayloadFromRequest(request),
    ).rejects.toThrow(HttpException);
  });

  it('authorized endpoints should be authorized with valid token', async () => {
    for (const endpoint of endpoints) {
      if (endpoint.method === 'post') {
        const response = await request(app.getHttpServer())
          .post(endpoint.url)
          .send(endpoint.body)
          .set('Authorization', `${validJwtToken}`)
          .type('application/json');

        expect(response.statusCode).not.toBe(HttpStatus.UNAUTHORIZED);
      } else if (endpoint.method === 'get') {
        const response = await request(app.getHttpServer())
          .get(endpoint.url)
          .query(endpoint.query || {})
          .set('Authorization', `${validJwtToken}`)
          .type('application/json');

        expect(response.statusCode).not.toBe(HttpStatus.UNAUTHORIZED);
      }
    }
  });

  it('authorized endpoints should be unauthorized with invalid token', async () => {
    for (const endpoint of endpoints) {
      if (endpoint.method === 'post') {
        const response = await request(app.getHttpServer())
          .post(endpoint.url)
          .send(endpoint.body)
          .set('Authorization', `${invalidJwtToken}`)
          .type('application/json');

        expect(response.statusCode).toBe(HttpStatus.UNAUTHORIZED);
      } else if (endpoint.method === 'get') {
        const response = await request(app.getHttpServer())
          .get(endpoint.url)
          .query(endpoint.query || {})
          .set('Authorization', `${invalidJwtToken}`)
          .type('application/json');

        expect(response.statusCode).toBe(HttpStatus.UNAUTHORIZED);
      }
    }
  });
});
